<?php


if (!defined('ABSPATH')) {
    exit;
}

/**
 * WC_Web_Payment_Gateway class.
 *
 * @since 2.0.0
 * @extends WC_Payment_Gateway
 */
class WC_Web_Payment_Gateway_Subscription extends WC_Web_Payment_Gateway
{

   
    public function __construct()
    {
        parent::__construct();

        $subscriptions = [
            'subscriptions',
            'subscription_cancellation',
            'subscription_suspension',
            'subscription_reactivation',
        ];

        $this->supports = array_merge($this->supports, $subscriptions);

        $this->apimethod['recurring'] = '/api/v1/payment/recurring';
        $this->apimethod['retry'] = '/api/v1/payment/retry';

        add_action('woocommerce_scheduled_subscription_payment_' . $this->id, [$this, 'scheduled_subscription_payment'], 10, 2);
    }

    /**
     * Process the payment and return the result.
     *
     * @param int @order_id
     * @return array
     */
    public function process_payment($order_id)
    {

        if (wcs_order_contains_subscription($order_id) || wcs_is_subscription($order_id)) {

            if ($this->iframe == 'yes') {
                return array(
                    'result' => 'success',
                    'redirect' => $this->get_return_url(wc_get_order($order_id)),
                );
            } else {
                return $this->process_subsscription($order_id);
            }
            
        }else{
            return parent::process_payment($order_id);
        }
      
    }

    private function process_subsscription($order_id)
    {
        $sub = wc_get_order($order_id);


        if ($this->settings['vat_calculation'] === 'yes') {
           
            $orderTotal = $sub->get_total();
            $orderTax = $sub->get_total_tax();

            $amountTaxFree = $orderTotal - $orderTax;
            $amount = $this->formatPrice($amountTaxFree);
        }else{
            $amount = $this->formatPrice($sub->get_total());
        }


        $billing_address = $this->buildBillingAddress($sub);
        $customer = $this->buildCustomer($sub);

        $orderJson = [
            'number' => "$order_id",
            'description' => __('Payment Order # ', 'woocommerce') . $order_id . __(' in the store ', 'woocommerce') . home_url('/'),
            'amount' => $amount,
            'currency' => get_woocommerce_currency(),
        ];


        $str_to_hash = $order_id . $amount . get_woocommerce_currency() . __('Payment Order # ', 'woocommerce') . $order_id . __(' in the store ', 'woocommerce') . home_url('/') . $this->password;
        $hash = sha1(md5(strtoupper($str_to_hash)));

        $mainJson = [
                'merchant_key' => $this->secret,
                'operation'    => 'purchase',
                'order'        => $orderJson,
                'customer'     => $customer,
                'billing_address' => $billing_address,
                'success_url' => $this->get_return_url($sub),
                'cancel_url'   => $sub->get_view_order_url(),
                'hash'         => $hash,

                'recurring_init' => true,
                'req_token ' => true,
                
        ];

        $methods = $this->method;
        if (!empty($methods)) {
             $main_json['methods'] = $methods;
        }

        if (!empty($this->additional_payment_parameters)) {
            $main_json['parameters'] = $this->buildAdditionalParameters($order_id);
        }

        if ($this->settings['vat_calculation'] === 'yes') {
            $mainJson['vat_calc'] = true;
        }

        if (!empty($this->settings['form_id'])) {
            $main_json['from_id'] = $this->settings['form_id'];
        }

        if ($this->iframe == 'yes') {
            if ($this->url_target != 'noused') {
                $main_json['url_target'] = '_' . $this->url_target;
            }
        }

        $action_adr = rtrim($this->url,'/').$this->apimethod['checkout'];
        
        $response = $this->makeRequest($action_adr, $mainJson);
        if ($response === false) {
            return false;
        }

        return [
            'result'   => 'success',
            'redirect' => $response['redirect_url'],
        ];
        
    }

    public function check_ipn_response()
    {

        $this->log('POST api: ' . print_r($_POST, true));
        $this->log('wpgLog', 'GET api: ' . print_r($_GET, true));

        if ($_SERVER['REQUEST_METHOD'] != 'POST') {
            exit;
        }

        $hash_string = $_POST['id'] . $_POST['order_number'] . $_POST['order_amount'] . $_POST['order_currency'] . $_POST['order_description'] . $this->password;
        $hash = sha1(md5(strtoupper($hash_string)));

        if ($_POST['hash'] != $hash) {
            exit;
        }

        $order_id = $_POST['order_number'];

        if (wcs_order_contains_subscription($order_id) || wcs_is_subscription($order_id)) {
            $this->processSubscriptionOrder();
        }else{
            $this->processSimpleOrder();
        }
                
    }

    private function processSubscriptionOrder()
    {
        global $woocommerce;

        $order = wc_get_order($_POST['order_number']);
        $order->set_transaction_id($_POST['id']);

        if ($order->get_status() == 'pending' || $order->get_status() == 'waiting' || $order->get_status() == 'failed' || $order->get_status() == '') {

            if ($_POST['status'] == 'success' && $_POST['type'] == 'sale') {
                //successful purchase
                $woocommerce->cart->empty_cart();
                $order->update_status('completed', 'Payment successfully paid'); //completed or processing
              
                // card_token
                $order->add_meta_data('wpg_recurring_init_trans_id', $_POST['recurring_init_trans_id']);
                $order->add_meta_data('wpg_recurring_token', $_POST['recurring_token']);
                
                if (isset($_POST['vat_amount'])) {
                    $order->add_meta_data('wpg_vat_amount', $_POST['vat_amount']);
                }
                $order->save();
                exit;
            }

            if ($_POST['status'] == 'waiting' && $_POST['type'] == 'sale') {
                //waiting purchase
                $order->update_status('on-hold', __('On hold', 'woocommerce'));
                exit;
            }

            if ($_POST['status'] == 'fail' && $_POST['type'] == 'sale') {
                //failed purchase
                $order->update_status('failed', $_POST['reason']);
                exit;
            }
        }

    }

    private function processSimpleOrder()
    {
        global $woocommerce;

        $order = wc_get_order($_POST['order_number']);

        $order->set_transaction_id($_POST['id']);

        if ($order->get_status() == 'pending' || $order->get_status() == 'waiting' || $order->get_status() == 'failed' || $order->get_status() == '') {

            if ($_POST['status'] == 'success' && $_POST['type'] == 'sale') {
                //successful purchase
                $woocommerce->cart->empty_cart();
                $order->update_status($this->order_status_after_payment, 'Payment successfully paid');

                if (isset($_POST['vat_amount'])) {
                    $order->add_meta_data('wpg_vat_amount', $_POST['vat_amount']);
                    $order->save();
                }

                exit;
            }

            if ($_POST['status'] == 'waiting' && $_POST['type'] == 'sale') {
                //waiting purchase
                $order->update_status('on-hold', __('On hold', 'woocommerce'));
                exit;
            }

            if ($_POST['status'] == 'fail' && $_POST['type'] == 'sale') {
                //failed purchase
                $order->update_status('failed', $_POST['reason']);
                exit;
            }
        }

        if ($order->get_status() == 'completed' || $order->get_status() == 'processing') {
            if ($_POST['status'] == 'success' && $_POST['type'] == 'refund') {
                $order->add_order_note('Refund confirmed by the payment system');
                exit;
            }
            if ($_POST['status'] == 'fail' && $_POST['type'] == 'refund') {
                $order->update_status('failed', $_POST['reason']);
                exit;
            }
        }

        die('default Order');
    }

    public function scheduled_subscription_payment($total, $renewal_order)
    {
        $subscriptions = wcs_get_subscriptions_for_renewal_order( $renewal_order->get_id() );
        $subscription = array_pop($subscriptions);
        $parentIdOrder = $subscription->get_parent_id();

        $this->log('Debug: ' . print_r([
            'total' => $total,
            'renewal_order' => $renewal_order->get_id(),
            'parent_order' => $parentIdOrder,
        ], true));

        $parentOrder = wc_get_order($parentIdOrder);

        $recurring_token = $parentOrder->get_meta('wpg_recurring_token');
        $recurring_init_trans_id = $parentOrder->get_meta('wpg_recurring_init_trans_id');

        $order_id = $renewal_order->get_id();

        $amount = $this->formatPrice($renewal_order->get_total());

        $order = [
            'number' => (string)$order_id,
            'amount' => $amount,
            'description' =>  __('Renewal subscription Order # ', 'woocommerce') . $order_id . __(' in the store ', 'woocommerce') . home_url('/')
        ];

        $str_to_hash = $recurring_init_trans_id . $recurring_token . $order['number'] . $order['amount'] . $order['description'] . $this->password;
        $hash = sha1(md5(strtoupper($str_to_hash)));

        $mainJson = [
            'merchant_key' => $this->secret,
            'recurring_init_trans_id' => $recurring_init_trans_id,
            'recurring_token' => $recurring_token,
            'order' => $order,
            'hash' => $hash,
        ];


        $action_adr = rtrim($this->url,'/').$this->apimethod['recurring'];
        $result = $this->makeRequest($action_adr, $mainJson);

        if ($result == false) {
             $renewal_order->add_order_note(__('Scheduled subscription payment failed. Couldn\'t connect to gateway server.', 'woocommerce-web-payment-gateway'));
        }

        if ($result['status'] == 'settled') {
            $renewal_order->add_order_note(__('Scheduled subscription payment completed.', 'woocommerce-web-payment-gateway') );
            WC_Subscriptions_Manager::process_subscription_payments_on_order($order_id);
        }elseif ($result['status'] == 'declined') {
            $renewal_order->add_order_note(__('Scheduled subscription payment failed. Error: ', 'woocommerce-web-payment-gateway') . $result['reason']);
            WC_Subscriptions_Manager::process_subscription_payment_failure_on_order($order_id);
        }

        $renewal_order->add_meta_data('wpg_paymnet_id', $result['payment_id']);
        $renewal_order->save();
    }

    public function receipt_page($order_or_id)
    {

        if ($order_or_id instanceof WC_Order) {
            $order = $order_or_id;
        }else{
            $order = new WC_Order($order_or_id);
        }

        if (!$order->is_paid() && !$this->checkIfIframeShowed($order)) {
            $url = $this->process_subsscription($order->get_id());

            WC()->session->set('wpg_showed_' . $order->get_id(), '1');

            $this->getFrame($url['redirect']);
        }
    }

}
